#include "SDL/SDL.h"
#include <stdio.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include "puzzle.h"
#include "sdlfont.h"

#define GRAPHIC_BPP 8

int resx=800,resy=600;                /*  resolution (window size) */
double aspectratio=1.0;               /*  >1=wider, 1=square <1=taller */
int thick=1;                          /*  thickness of grid line */
int thin=1;                           /*  thickness of thinner grid line */
static int fullscreen=0;              /*  1=fullscreen 0=windowed */

/*  special overriding options for specific games */
int heyawakethick=-1;

uchar unfilledcol=GRAY8;
uchar mustprocesscol=GRAYBLUE8;
uchar blankcol=WHITE8;
uchar filledcol=BLACK8;
uchar filled2col=BLUE8;
uchar okcol=GREEN8;
uchar almostokcol=YELLOW8;
uchar errorcol=RED8;
uchar darkerrorcol=DARKRED8;
uchar darkererrorcol=DARKERRED8;
uchar lightcol=GREEN8;

/*  control schemes for the different games */
int controlscheme_nurikabe=0;
int controlscheme_akari=0;
int controlscheme_heyawake=0;
int controlscheme_hitori=0;
int controlscheme_picross=0;

/*  keyboard settings */

int undokey=SDLK_BACKSPACE;
int hintkey=SDLK_h;
int verifykey=SDLK_v;

static int videoflags;

SDL_Surface *screen;
sdl_font *font;

int width,height;                     /*  size of cell (including line) */
int startx,starty;                    /*  top left of grid area */

static int bpp;                       /*  _bytes_ per pixel */
static int pitch;                     /*  bytes per scanline */

int keys[512];

/*  length of a constant string */
#define GRAPHIC_FS 1024
static char s[GRAPHIC_FS];
static char t[GRAPHIC_FS];
static char u[GRAPHIC_FS];

/*  let's supply dx/dy arrays */
int dx[]={1,0,-1,0,0},dy[]={0,1,0,-1,0};
int dx8[]={1,0,-1,0,1,1,-1,-1},dy8[]={0,1,0,-1,1,-1,1,-1};

void error(char *s,...) {
  static char t[GRAPHIC_FS];
  va_list argptr;
  va_start(argptr,s);
  vsprintf(t,s,argptr);
  va_end(argptr);
  /*  TODO draw to screen, wait for keypress and exit */
  /*  for now, just output to a file */
  {
    FILE *f=fopen("error.txt","w");
    fputs(t,f);
    fclose(f);
  }
  exit(1);
}

void resetlog() {
  FILE *f=fopen("log.txt","w");
  fclose(f);
}

void logprintf(char *s,...) {
  static char t[1000];
  FILE *f;
  va_list argptr;
  va_start(argptr,s);
  vsprintf(t,s,argptr);
  va_end(argptr);
  f=fopen("log.txt","a");
  fprintf(f,"%s",t);
  fclose(f);
}

static void split(char *s,char *t,char *u) {
  char *q=strchr(s,'=');
  if(!q) {
    t[0]=u[0]=0;
    return;
  }
  while(isspace(*s)) s++;
  while(isalnum(*s) || *s=='_') *t++=*s++;
  s=q+1;
  while(isspace(*s)) s++;
  while(isalnum(*s) || *s=='.' || *s=='_') *u++=*s++;
  *t=*u=0;
}

static uchar parsecolour(char *s) {
  int v=0,i;
  for(i=0;s[i];i++) if(s[i]>='0' && s[i]<='9') v=v*10+s[i]-'0';
  else if(!strcmp(s,"BLACK")) return BLACK8;
  else if(!strcmp(s,"GRAY")) return GRAY8;
  else if(!strcmp(s,"GRAYBLUE")) return GRAYBLUE8;
  else if(!strcmp(s,"WHITE")) return WHITE8;
  else if(!strcmp(s,"GREEN")) return GREEN8;
  else if(!strcmp(s,"YELLOW")) return YELLOW8;
  else if(!strcmp(s,"RED")) return RED8;
  else if(!strcmp(s,"DARKRED")) return DARKRED8;
  else if(!strcmp(s,"DARKERRED")) return DARKERRED8;
  else if(!strcmp(s,"DARKBLUE")) return DARKBLUE8;
  else if(!strcmp(s,"BLUE")) return BLUE8;
  else error("illegal colour in definition file: %s\n",s);
  return (uchar)v;
}

static void initinifile() {
  FILE *f=fopen("puzzle.ini","r");
  if(!f) return;
  while(fgets(s,GRAPHIC_FS,f)) if(s[0]!='%') {
    split(s,t,u);
    if(!strcmp(t,"x")) resx=strtol(u,0,10);
    else if(!strcmp(t,"y")) resy=strtol(u,0,10);
    else if(!strcmp(t,"fullscreen")) fullscreen=strtol(u,0,10);
    else if(!strcmp(t,"aspectratio")) aspectratio=strtod(u,0);
    else if(!strcmp(t,"thick")) thick=strtol(u,0,10);
    else if(!strcmp(t,"thin")) thin=strtol(u,0,10);
    /*  game specific other options */
    else if(!strcmp(t,"heyawakethick")) heyawakethick=strtol(u,0,10);
    /*  control schemes */
    else if(!strcmp(t,"controlscheme_nurikabe")) controlscheme_nurikabe=strtol(u,0,10);
    else if(!strcmp(t,"controlscheme_akari")) controlscheme_akari=strtol(u,0,10);
    else if(!strcmp(t,"controlscheme_heyawake")) controlscheme_heyawake=strtol(u,0,10);
    else if(!strcmp(t,"controlscheme_hitori")) controlscheme_hitori=strtol(u,0,10);
    else if(!strcmp(t,"controlscheme_picross")) controlscheme_picross=strtol(u,0,10);
    /*  colour options */
    else if(!strcmp(t,"unfilledcol")) unfilledcol=parsecolour(u);
    else if(!strcmp(t,"mustprocesscol")) mustprocesscol=parsecolour(u);
    else if(!strcmp(t,"filledcol")) filledcol=parsecolour(u);
    else if(!strcmp(t,"filled2col")) filled2col=parsecolour(u);
    else if(!strcmp(t,"blankcol")) blankcol=parsecolour(u);
    else if(!strcmp(t,"okcol")) okcol=parsecolour(u);
    else if(!strcmp(t,"almostokcol")) almostokcol=parsecolour(u);
    else if(!strcmp(t,"errorcol")) errorcol=parsecolour(u);
    else if(!strcmp(t,"darkerrorcol")) darkerrorcol=parsecolour(u);
    else if(!strcmp(t,"darkererrorcol")) darkererrorcol=parsecolour(u);
    else if(!strcmp(t,"lightcol")) lightcol=parsecolour(u);
  }
  fclose(f);
}

static void updatevideoinfo() {
  pitch=screen->pitch;
  bpp=screen->format->BytesPerPixel;
}

static void initvideo() {
  /*  settings taken from tile world. what is SDL_ANYFORMAT? */
  if(SDL_Init(SDL_INIT_VIDEO)<0) exit(1);
  videoflags=SDL_SWSURFACE | SDL_ANYFORMAT;
  videoflags|=SDL_RESIZABLE;
  if(fullscreen) videoflags|=SDL_FULLSCREEN;
  if(!(screen=SDL_SetVideoMode(resx,resy,GRAPHIC_BPP,videoflags))) exit(1);
  updatevideoinfo();
}

void initgr() {
  initinifile();
  initvideo();
  memset(keys,0,sizeof(keys));
  if(!(font=sdl_font_load("font.bmp"))) error("error loading font.");
}

void shutdown() {
  sdl_font_free(font);
  SDL_Quit();
}

/*  draw a horizontal line 1 pixel thick */
void drawhorizontalline8(int x1,int x2,int y,Uint8 col) {
  int w=x2-x1+1,i;
  Uint8 *p=(Uint8 *)screen->pixels+y*pitch+x1*bpp;
  for(i=0;i<w;i++) p[i*bpp]=col;
}

/*  draw a vertical line 1 pixel thick */
void drawverticalline8(int x,int y1,int y2,Uint8 col) {
  int i;
  Uint8 *p=(Uint8 *)screen->pixels+x*bpp+y1*pitch;
  for(i=y1;i<=y2;i++) *p=col,p+=pitch;
}

/*  draw a rectangle */
void drawrectangle8(int x1,int y1,int x2,int y2,Uint8 col) {
  int i,w=x2-x1+1,j;
  Uint8 *p=(Uint8 *)screen->pixels+x1*bpp+y1*pitch;
  /*  TODO investigate performance for small widths */
  for(i=y1;i<=y2;i++) {
    for(j=0;j<w;j++) p[j*bpp]=col;
    p+=pitch;
  }
}

/*  draw a filled cell in cell x,y with custom border */
void drawsolidcell8w(int u,int v,Uint8 col,int left,int up,int right,int down) {
  drawrectangle8(startx+u*width+left,starty+v*height+up,
    startx+(u+1)*width-1-right,starty+(v+1)*height-1-down,col);
}

/*  draw a filled cell in cell x,y */
void drawsolidcell8(int u,int v,Uint8 col) {
  drawsolidcell8w(u,v,col,thick,thick,0,0);
}

/*  draw a number in cell x,y with custom border */
void drawnumbercell8w(int u,int v,int num,Uint8 col,Uint8 col2,Uint8 b,int left,int up,int right,int down) {
  drawrectangle8(startx+u*width+left,starty+v*height+up,
    startx+(u+1)*width-1-right,starty+(v+1)*height-1-down,b);
  sdl_font_printf(screen,font,startx+u*width+left+1,starty+v*height+up+1,col,
    col2,"%d",num);
}

/*  draw a number in cell x,y */
void drawnumbercell8(int u,int v,int num,Uint8 col,Uint8 col2,Uint8 b) {
  drawnumbercell8w(u,v,num,col,col2,b,thick,thick,0,0);
}

/*  draw a circular disc in cell x,y with radius r */
void drawdisc(int u,int v,double r,Uint8 col,Uint8 b) {
  double cx=thick+(width-thick)*.5,cy=thick+(height-thick)*.5;
  double dx,dy;
  int i,j;
  Uint8 *p;
  r*=r;
  for(j=thick;j<height;j++) {
    p=(Uint8 *)screen->pixels+(u*width+startx)*bpp+(v*height+starty+j)*pitch;
    for(i=thick;i<width;i++) {
      dx=i-cx; dy=j-cy;
      p[i*bpp]=dx*dx+dy*dy<=r?col:b;
    }
  }
}

/*  clear the screen to one colour */
void clear8(Uint8 col) {
  int j,i;
  Uint8 *p;
  for(j=0;j<resy;j++) {
    p=(Uint8 *)screen->pixels+j*pitch;
    for(i=0;i<resx;i++) p[i*bpp]=col;
  }
  SDL_UpdateRect(screen,0,0,resx,resy);
}

/*  display the palette on the screen in the following manner:
    0 1 .. 15, 16 17 .. 31, ..., 240 241 .. 255
*/
void palette8() {
  int i,j;
  Uint8 *p;
  for(j=0;j<resy;j++) {
    p=(Uint8 *)screen->pixels+j*pitch;
    for(i=0;i<resx;i++) p[i*bpp]=j*16/resy*16+i*16/resx;
  }
}

/*  determine the size of each cell */
void updatescale(int resx,int resy,int x,int y,int thick) {
  width=(resx-thick-1)/x;
  height=(resy-thick-1)/y;
  if((int)(width/aspectratio)<height) height=(int)(width/aspectratio);
  if((int)(width/aspectratio)>height) width=(int)(height*aspectratio);
}

/*   return cell coordinates underneath mouse coordinates */
void getcell(int mousex,int mousey,int *cellx,int *celly) {
  *cellx=*celly=-1;
  if(mousex<startx || mousey<starty) return;
  /*  TODO exit if mouse pointer is to the right/below */
  /*  TODO exit if mouse pointer is exactly on the grid */
  *cellx=(mousex-startx)/width;
  *celly=(mousey-starty)/height;
}

int manhattandist(int x1,int y1,int x2,int y2) {
  int dx=x1-x2,dy=y1-y2;
  if(dx<0) dx=-dx;
  if(dy<0) dy=-dy;
  return dx+dy;
}

/*  refresh the screen under given cell coordinates */
void sdlrefreshcell(int cellx,int celly) {
  SDL_UpdateRect(screen,startx+cellx*width,starty+celly*height,width,height);
}

/*  rudimentary stack routines */
#define MAXSTACK 1000000
static int stack[MAXSTACK];
static int stackpos;

void stackpush(int val) {
  if(stackpos==MAXSTACK) error("attempt to push on a full stack!\n");
  stack[stackpos++]=val;
}

/*  return true if stack is empty */
int stackempty() {
  return !stackpos;
}

int stackpop() {
  if(!stackpos) error("attempt to pop from an empty stack!\n");
  return stack[--stackpos];
}

int getstackpos() {
  return stackpos;
}

void setstackpos(int sp) {
  stackpos=sp;
}

/*  loop until an event happens, and return it
    0-511:    keydown + id
    512-1023: keyup + id
    1024:    press mouse button (+ global variables)
    2048:    release mouse button (+ global variables)
    65536:    resize event
    131072:   close program event
*/

int event_mousebutton;
int event_mousex;
int event_mousey;
static SDL_Event event;

int getevent() {
  while(SDL_WaitEvent(&event)) {
    switch(event.type) {
    case SDL_VIDEORESIZE:
      screen=SDL_SetVideoMode(resx=event.resize.w,resy=event.resize.h,GRAPHIC_BPP,videoflags);
      if(!screen) exit(1);
      updatevideoinfo();
      return EVENT_RESIZE;
    case SDL_KEYDOWN:
      keys[event.key.keysym.sym]=1;
      return event.key.keysym.sym+EVENT_KEYDOWN;
    case SDL_KEYUP:
      keys[event.key.keysym.sym]=0;
      return event.key.keysym.sym+EVENT_KEYUP;
    case SDL_MOUSEBUTTONDOWN:
      event_mousebutton=event.button.button;
      event_mousex=event.button.x;
      event_mousey=event.button.y;
      return EVENT_MOUSEDOWN;
    case SDL_MOUSEBUTTONUP:
      event_mousebutton=event.button.button;
      event_mousex=event.button.x;
      event_mousey=event.button.y;
      return EVENT_MOUSEUP;
      /*  TODO return mousemotion event */
      /* EXAMPLE */
/*
            case SDL_MOUSEMOTION:
                printf("Mouse moved by %d,%d to (%d,%d)\n", 
                       event.motion.xrel, event.motion.yrel,
                       event.motion.x, event.motion.y);
                break;
*/
      case SDL_QUIT:
      return EVENT_QUIT;
    }
  }
  return EVENT_NOEVENT;
}

/*  wait for a keypress or mouse key press */
/*  TODO resizing will not force redraw while we're here */
void anykeypress() {
  int ievent;
  while(1) {
    ievent=getevent();
    if(ievent>=EVENT_KEYDOWN && ievent<EVENT_KEYUP) break;
    if(ievent==EVENT_MOUSEDOWN) break;
    if(ievent==EVENT_QUIT) {
      /*  push back quit event  */
      SDL_PushEvent(&event);
      break;
    }
  }
}

/*  TODO make a more advanced messagebox with support for multiple lines */
#define RAMME 16
#define SDL_FONT_S 65536
void messagebox(char *fmt,...) {
  static char t[SDL_FONT_S];
  int w,i,j,x,y,h=font->height,R=RAMME+RAMME;
  uchar *backup;
  Uint8 *p;
  va_list argptr;
  va_start(argptr,fmt);
  vsprintf(t,fmt,argptr);
  va_end(argptr);
  w=sdl_font_width(font,t);
  backup=(uchar *)malloc((w+R)*(h+R));
  x=(resx-w-R)/2;
  y=(resy-font->height-R)/2;
  if(SDL_MUSTLOCK(screen)) SDL_LockSurface(screen);
  for(i=0;i<font->height+R;i++) {
    p=(Uint8 *)screen->pixels+(i+y)*pitch+x*bpp;
    for(j=0;j<w+R;j++) backup[(w+R)*i+j]=p[j*bpp];
  }
  /*  green interior */
  drawrectangle8(x,y,x+w+R-1,y+h+R-1,GREEN8);
  /*  big white outline */
  drawrectangle8(x,y,x+w+R-1,y+3,WHITE8);
  drawrectangle8(x,y+h+R-4,x+w+R-1,y+h+R-1,WHITE8);
  drawrectangle8(x,y,x+3,y+h+R-1,WHITE8);
  drawrectangle8(x+w+R-3,y,x+w+R-1,y+h+R-1,WHITE8);
  /*  tiny black outline */
  drawhorizontalline8(x,x+w+R-1,y,BLACK8);
  drawhorizontalline8(x,x+w+R-1,y+h+R-1,BLACK8);
  drawverticalline8(x,y,y+h+R-1,BLACK8);
  drawverticalline8(x+w+R-1,y,y+h+R-1,BLACK8);
  sdl_font_printf(screen,font,x+RAMME,y+RAMME,WHITE8,WHITE8,t);
  if(SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
  SDL_UpdateRect(screen,x,y,w+R,h+R);
  anykeypress();
  if(SDL_MUSTLOCK(screen)) SDL_LockSurface(screen);
  for(i=0;i<font->height+R;i++) {
    p=(Uint8 *)screen->pixels+(i+y)*pitch+x*bpp;
    for(j=0;j<w+R;j++) p[j*bpp]=backup[(w+R)*i+j];
  }
  if(SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
  SDL_UpdateRect(screen,x,y,w+R,h+R);
  free(backup);
}
