#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "SDL/SDL.h"
#include "puzzle.h"

/*  max map size */
#define MAXS 128
/*  max string length */
#define MAXSTR 1024
static int x,y;                 /*  map size */
static char difficulty[MAXSTR]; /*  string holding the difficulty */

static int dummy;               /*  dummy variable */

/*  the actual content of a cell resides within [thick,width)x[thick,height).
    hence, the entire draw ares is [0,width*x+thick)x[0,height*y+thick),
    before translation. */
/*  the actual variables width,height are stored in graphics.c */

/*  NB, we must have UNFILLED<BLOCKED<EMPTY<1 (numbered cell) */
#define UNFILLED    -100
#define BLOCKED     -1
#define EMPTY       0

/*  internal format:
    -100: unfilled cell
    -1: blocked cell (water, wall)
    0:  empty cell (island, room)
    1-  numbered cell (part of island)
*/
static int m[MAXS][MAXS];
/*  additional info:
    0:  nothing special (island too small)
    1:  island is complete, but need to be closed
    2:  island is complete
    3:  island is too large, or island is closed with wrong amount,
        or island has 0 or 2 numbered cells
*/
#define ISLAND_TOOSMALL 0
#define ISLAND_TOCLOSE  1
#define ISLAND_GOOD     2
#define ISLAND_TOOLARGE 3
static int st[MAXS][MAXS];
static char touched[MAXS][MAXS];  /*  1 if cell is changed and need to be redrawn */

static int totalnum;              /*  sum of numbered cells = number of blanks */
/*  also: x*y-totalnum: required number of walls */

/*  move queue for hint system */
#define MAXMQ MAXS*MAXS*3
static int mq[MAXMQ];
static int mqs,mqe;

static int convchar(char *s,int *i) {
  char c=s[(*i)++];
  int v;
  if(!c) error("string in level definition ended prematurely.");
  else if(c=='.') return UNFILLED;
  else if(c>='1' && c<='9') return c-48;
  else if(c>='a' && c<='z') return c-'a'+10;
  else if(c>='A' && c<='Z') return c-'A'+36;
  else if(c=='{') {
    v=0;
    while(s[*i] && s[*i]!='}') v=v*10+s[(*i)++]-48;
    if(s[*i]=='}') (*i)++;
    return v;
  }
  error("invalid character %c in level definition.",c);
  return 0;
}

/*  use this as a model for loadpuzzle for other puzzles */
/*  TODO abstrahate this? */
static void loadpuzzle(char *path) {
  static char s[MAXSTR];
  FILE *f=fopen(path,"r");
  int z=0,ln=0,i,j;
  if(!f) error("couldn't open the file %s\n",path);
  while(fgets(s,MAXSTR,f)) if(s[0]!='%') {
    switch(z) {
    case 1:
      strcpy(difficulty,s);
    case 0:
      z++;
      break;
    case 2:
      sscanf(s,"%d %d",&x,&y);
      z++;
      break;
    case 3:
      for(i=j=0;j<x;j++) m[j][ln]=convchar(s,&i);
      ln++;
    }
  }
  fclose(f);
  /*  set top left of grid area in window */
  startx=10,starty=30;
  totalnum=mqs=mqe=0;
  for(i=0;i<x;i++) for(j=0;j<y;j++) {
    st[i][j]=touched[i][j]=0;
    if(m[i][j]>0) totalnum+=m[i][j];
  }
}

static void updatecell(int u,int v) {
  /*  undetermined - white */
  if(m[u][v]==UNFILLED) drawsolidcell8(u,v,unfilledcol);
  /*  water, wall */
  else if(m[u][v]==BLOCKED) drawsolidcell8(u,v,filledcol);
  else if(m[u][v]==EMPTY) {
    /*  unnumbered island, room */
    if(st[u][v]==ISLAND_TOOSMALL) drawsolidcell8(u,v,blankcol);
    else if(st[u][v]==ISLAND_TOCLOSE) drawsolidcell8(u,v,almostokcol);
    else if(st[u][v]==ISLAND_GOOD) drawsolidcell8(u,v,okcol);
    else if(st[u][v]==ISLAND_TOOLARGE) drawsolidcell8(u,v,errorcol);
  } else if(m[u][v]>0) {
    /*  numbered island, room */
    if(st[u][v]==ISLAND_TOOSMALL) drawnumbercell8(u,v,m[u][v],BLACK8,BLACK8,blankcol);
    else if(st[u][v]==ISLAND_TOCLOSE) drawnumbercell8(u,v,m[u][v],BLACK8,BLACK8,almostokcol);
    else if(st[u][v]==ISLAND_GOOD) drawnumbercell8(u,v,m[u][v],BLACK8,BLACK8,okcol);
    else if(st[u][v]==ISLAND_TOOLARGE) drawnumbercell8(u,v,m[u][v],BLACK8,BLACK8,errorcol);
  }
}

static void drawgrid() {
  int i,j;
  if(SDL_MUSTLOCK(screen)) SDL_LockSurface(screen);
  clear8(WHITE8);
  updatescale(resx-startx,resy-starty,x,y,thick);
  if(thick) {
    for(i=0;i<=y;i++) for(j=0;j<thick;j++) drawhorizontalline8(startx,startx+width*x+thick-1,starty+i*height+j,BLACK8);
    for(i=0;i<=x;i++) drawrectangle8(startx+width*i,starty,startx+i*width+thick-1,starty+y*height+thick-1,BLACK8);
  }
  for(i=0;i<x;i++) for(j=0;j<y;j++) updatecell(i,j);
  if(SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
  SDL_UpdateRect(screen,0,0,resx,resy);
}

static int togglecell(int val) {
  switch(val) {
  case UNFILLED:
  case EMPTY:
    return BLOCKED;
  case BLOCKED:
    return EMPTY;
  default:
    error("board hath illegal values.");
    return 0;
  }
}

/*  TODO oh noes, here we loop through the entire board.
    if this ever turns out to be a performance problem, make
    a list of touched cells instead */
static void partialredraw() {
  int i,j;
  if(SDL_MUSTLOCK(screen)) SDL_LockSurface(screen);
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(touched[i][j]) updatecell(i,j);
  if(SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(touched[i][j]) {
    sdlrefreshcell(i,j);
    touched[i][j]=0;
  }
}

/*  here starts routines to recalculate the fancy background colours for
    the cells. for each change, examine the neighbourhood of the change
    and determine if regions contains enough cells, too many cells, or
    too many numbered cells. */

/*  nice general purpose bfs variables (for completeness check and hint system) */
static uchar visit[MAXS][MAXS];
static int qs,qe,q[MAXS*MAXS*2];

/*  only needs to be called once */
static void initcompleteness() {
  memset(visit,0,sizeof(visit));
  qs=qe=0;
}

/*  type:
    0: search for empty, numbered
    1: search for blocked
    2: search for empty, numbered, unfilled
    3: search for blocked, unfilled
    return: tot: number of cells examined
            num: numbered cell (-1:more than one, >0: the number, 0: no numbered cell
            islands: number of island cells found
*/
static void genericbfs(int sx,int sy,int type,int *tot,int *num,int *islands) {
  int cx,cy,x2,y2,i;
  *tot=*num=*islands=0;
  if(visit[sx][sy]) return;
  if(type==0 && m[sx][sy]<EMPTY) return;
  if(type==1 && m[sx][sy]!=BLOCKED) return;
  if(type==2 && m[sx][sy]<EMPTY && m[sx][sy]!=UNFILLED) return;
  if(type==3 && m[sx][sy]!=BLOCKED && m[sx][sy]!=UNFILLED) return;
  q[qe++]=sx,q[qe++]=sy;
  visit[sx][sy]=1;
  *tot=1;
  if(m[sx][sy]>0) (*islands)++;
  *num=m[sx][sy]>0?m[sx][sy]:0;
  while(qs<qe) {
    cx=q[qs++],cy=q[qs++];
    for(i=0;i<4;i++) {
      x2=cx+dx[i],y2=cy+dy[i];
      if(x2<0 || y2<0 || x2>=x || y2>=y || visit[x2][y2]) continue;
      if(type==0 && m[x2][y2]<EMPTY) continue;
      if(type==1 && m[x2][y2]!=BLOCKED) continue;
      if(type==2 && m[x2][y2]<EMPTY && m[x2][y2]!=UNFILLED) continue;
      if(type==3 && m[x2][y2]!=BLOCKED && m[x2][y2]!=UNFILLED) continue;
      (*tot)++;
      if(m[x2][y2]>=EMPTY) (*islands)++;
      if(m[x2][y2]>0) {
        if(!*num) *num=m[x2][y2];
        else *num=-1;
      }
      q[qe++]=x2,q[qe++]=y2;
      visit[x2][y2]=1;
    }
  }
}

/*  check region completeness from cellx,celly */
/*  WARNING, call cleanupbfs() after this */
static void recalccompleteness(int cellx,int celly,int visible) {
  int i,j,x2,y2,qs2=qe,ss=0,closed=1;
  int tot=1;    /*  # connected blank cells */
  int num=0;    /*  0: no number in region, >0: number encountered in region,
                    -1: more than one numbered cell in region */
  if(cellx<0 || celly<0 || cellx>=x || celly>=y || m[cellx][celly]==BLOCKED
    || m[cellx][celly]==UNFILLED || visit[cellx][celly]) return;
  genericbfs(cellx,celly,0,&tot,&num,&dummy);
  /*  check if region is enclosed */
  for(i=qs2;i<qe;i+=2) for(j=0;j<4;j++) {
    x2=q[i]+dx[j],y2=q[i+1]+dy[j];
    if(x2>=0 && y2>=0 && x2<x && y2<y && m[x2][y2]==UNFILLED) {
      closed=0;
      goto found;
    }
  }
found:
  /*  region is found, update colours */
  if(closed) {
    if(num<1 || num!=tot) ss=ISLAND_TOOLARGE;
    else ss=ISLAND_GOOD;
  } else {
    if(num<0 || (num>0 && tot>num)) ss=ISLAND_TOOLARGE;
    else if(num>0 && tot==num) ss=ISLAND_TOCLOSE;
    else ss=ISLAND_TOOSMALL;
  }
  for(i=qs2;i<qe;i+=2) {
    x2=q[i],y2=q[i+1];
    if(st[x2][y2]!=ss) st[x2][y2]=ss,touched[x2][y2]=visible;
  }
}

/*  clean up the data structures, clear visited and reset queue pos
    (must be done after a series of bfs-es */
static void cleanupbfs() {
  while(qe) visit[q[qe-2]][q[qe-1]]=0,qe-=2;
  qs=0;
}

/*  do move bookkeeping, including putting it on the stack */
static void domove(int cellx,int celly,int val) {
  if(val==m[cellx][celly]) error("logical error, tried to set cell to existing value");
  stackpush(cellx); stackpush(celly); stackpush(m[cellx][celly]);
  m[cellx][celly]=val;
    touched[cellx][celly]=1;
}

static void updatetoscreen(int cellx,int celly,int visible) {
  int i;
  for(i=0;i<5;i++) recalccompleteness(cellx+dx[i],celly+dy[i],visible);
  cleanupbfs();
  if(visible) partialredraw();
}

/*  TODO maybe find a better way to organize the following code */
/*  change board according to mouse click */
static void processmousedown() {
  int cellx,celly,v=controlscheme_nurikabe,up=0;
  if(event_mousebutton==SDL_BUTTON_LEFT) {
    getcell(event_mousex,event_mousey,&cellx,&celly);
    if(cellx<0 || celly<0 || cellx>=x || celly>=y || m[cellx][celly]>0) return;
    if(!v) {
      domove(cellx,celly,togglecell(m[cellx][celly])); up=1;
    } else if(v==1 && m[cellx][celly]!=EMPTY) {
      domove(cellx,celly,EMPTY); up=1;
    } else if(v==2 && m[cellx][celly]!=BLOCKED) {
      domove(cellx,celly,BLOCKED); up=1;
    }
  } else if(event_mousebutton==SDL_BUTTON_RIGHT) {
    getcell(event_mousex,event_mousey,&cellx,&celly);
    if(cellx<0 || celly<0 || cellx>=x || celly>=y || m[cellx][celly]>0) return;
    if(!v && m[cellx][celly]!=UNFILLED) {
      domove(cellx,celly,UNFILLED); up=1;
    } else if(v==1 && m[cellx][celly]!=BLOCKED) {
      domove(cellx,celly,BLOCKED); up=1;
    } else if(v==2 && m[cellx][celly]!=EMPTY) {
      domove(cellx,celly,EMPTY); up=1;
    }
  } else if(event_mousebutton==SDL_BUTTON_MIDDLE) {
    getcell(event_mousex,event_mousey,&cellx,&celly);
    if(cellx<0 || celly<0 || cellx>=x || celly>=y || m[cellx][celly]>0) return;
    if(v && m[cellx][celly]!=UNFILLED) {
      domove(cellx,celly,UNFILLED); up=1;
    }
  }
  if(up) updatetoscreen(cellx,celly,1);
}

static void undo() {
  if(!stackempty()) {
    int val=stackpop(),celly=stackpop(),cellx=stackpop();
    m[cellx][celly]=val;
    touched[cellx][celly]=1;
    updatetoscreen(cellx,celly,1);
  }
}

/*  hint system! */
/*  warning, this is not very efficient */

static void addmovetoqueue(int cellx,int celly,int val) {
  mq[mqe++]=cellx; mq[mqe++]=celly; mq[mqe++]=val;
  if(mqe==MAXMQ) mqe=0;
}

static int movequeueisempty() {
  return mqs==mqe;
}

/*  return:
    0: no moves in queue
    1: move successfully executed */
static int executeonemovefromqueue(int visible) {
loop:
  if(movequeueisempty()) return 0;
  /*  the hint system can produce some moves twice, don't redo moves */
  if(m[mq[mqs]][mq[mqs+1]]==mq[mqs+2]) {
    mqs+=3; if(mqs==MAXMQ) mqs=0;
    goto loop;
  }
  domove(mq[mqs],mq[mqs+1],mq[mqs+2]);
  updatetoscreen(mq[mqs],mq[mqs+1],visible);
  mqs+=3; if(mqs==MAXMQ) mqs=0;
  return 1;
}

static void executemovequeue() {
  while(executeonemovefromqueue(1));
}

static void silentexecutemovequeue() {
  while(executeonemovefromqueue(0));
}

/*  another helper array for hint */
#define MAXISLAND MAXS*MAXS
static int islandmap[MAXS][MAXS];
static int islandsize[MAXISLAND],islandid,islandidnum;
static int islandnum[MAXISLAND];

/*  helper routine: mark all islands with id */
/*  islandmap[x][y] is set to -1 if no island, otherwise an id
    islandsize[id] indicates the size of island number id
    islandnum[id] indicates the numbered cell within island number id
    islandid is the number of ids
    islandidnum is the number of islands with numbered cells
    the id for an island is >=0, <islandidnum if it contains a number
    it is >=islandidnum if it hasn't */
static void markallislandswithid() {
  int i,j,k,qs2,num,tot;
  for(islandid=i=0;i<x;i++) for(j=0;j<y;j++) islandmap[i][j]=-1;
  /*  mark all numbered islands */
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(islandmap[i][j]==-1 && m[i][j]>0) {
    qs2=qe;
    genericbfs(i,j,0,&tot,&num,&dummy);
    for(k=qs2;k<qe;k+=2) islandmap[q[k]][q[k+1]]=islandid;
    islandsize[islandid]=tot;
    islandnum[islandid++]=num;
  }
  islandidnum=islandid;
  /*  mark all nonnumbered islands */
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(islandmap[i][j]<0 && m[i][j]==EMPTY) {
    qs2=qe;
    genericbfs(i,j,0,&tot,&num,&dummy);
    for(k=qs2;k<qe;k+=2) islandmap[q[k]][q[k+1]]=islandid;
    islandsize[islandid]=tot;
    islandnum[islandid++]=0;
  }
  cleanupbfs();
}

/*  return 1 if solved, 0 if unsolved, -1 if error somewhere */
/*  assume that st is correctly filled in with island sizes! */
/*  TODO rewrite this so it doesn't depend on st[][]? should then be
    easier to use as subroutine in solver */
static int verifyboard() {
  int i,j,tot,num;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]>=EMPTY && st[i][j]==ISLAND_TOOLARGE) return -1;
  for(i=0;i<x-1;i++) for(j=0;j<y-1;j++) if(m[i][j]==BLOCKED && m[i+1][j]==BLOCKED && m[i][j+1]==BLOCKED && m[i+1][j+1]==BLOCKED) return -1;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==UNFILLED) return 0;
  /*  ok, no island errors and everything is filled in.
      check if blocked is connected */
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==BLOCKED) {
    genericbfs(i,j,1,&tot,&num,&dummy);
    cleanupbfs();
    if(tot!=x*y-totalnum) return -1;
  }
  return 1;
}

/*  deeper verify:
    - check if walls cannot be connected
    - check if there are islands that cannot grow to their full size
    TODO check if there exists unnumbered islands which cannot be
         connected to any existing numbered island. this heuristic
         will improve contradiction
    TODO check if there exists an enclosure with m islands, n squares,
         sum of islands k such that k-m-1>n
*/
static int deepverifyboard() {
  int r=verifyboard(),i,j,left,cx,cy,k,x2,y2,l,x3,y3,tot,num;
  if(r) return r;
  /*  there are unfilled cells. check if walls can be connected */
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==BLOCKED) {
    genericbfs(i,j,3,&tot,&num,&dummy);
    for(k=0;k<x;k++) for(l=0;l<y;l++) if(m[k][l]==BLOCKED && !visit[k][l]) {
      cleanupbfs();
      return -1;
    }
    cleanupbfs();
    goto breakout;
  }
breakout:
  markallislandswithid();
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]>EMPTY) {
    left=m[i][j]-1;
    q[qe++]=i; q[qe++]=j;
    visit[i][j]=1;
    while(qs<qe) {
      cx=q[qs++]; cy=q[qs++];
      for(k=0;k<4;k++) {
        x2=cx+dx[k],y2=cy+dy[k];
        if(x2<0 || y2<0 || x2>=x || y2>=y || visit[x2][y2] || m[x2][y2]<UNFILLED) continue;
        /*  check if cell is adjacent to another island */
        for(l=0;l<4;l++) {
          x3=x2+dx[l],y3=y2+dy[l];
          if(x3<0 || y3<0 || x3>=x || y3>=y || visit[x3][y3] || m[x3][y3]<EMPTY) continue;
          if(islandmap[x3][y3]>-1 && islandmap[x2][y2]!=islandmap[x3][y3] && islandmap[x3][y3]<islandidnum) goto nomove;
        }
        if(left>0) q[qe++]=x2,q[qe++]=y2,left--,visit[x2][y2]=1;
      nomove:;
      }
    }
    cleanupbfs();
/*    printf("island at %d,%d, size %d, %d left\n",i,j,m[i][j],left);*/
    if(left) return -1;
  }
  return 0;
}

static int level1surround1() {
  int i,j,k,x2,y2,ok=0;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==1) {
    for(k=0;k<4;k++) {
      x2=i+dx[k],y2=j+dy[k];
      if(x2<0 || y2<0 || x2>=x || y2>=y || m[x2][y2]!=UNFILLED) continue;
      addmovetoqueue(x2,y2,BLOCKED);
      ok=1;
    }
  }
  return ok;
}

static int level12x2() {
  int i,j,k,l,count,x2=0,y2=0,ok=0;
  for(i=0;i<x-1;i++) for(j=0;j<y-1;j++) {
    for(count=k=0;k<2;k++) for(l=0;l<2;l++) if(m[i+k][j+l]==BLOCKED) count++;
    else x2=i+k,y2=j+l;
    if(count==3 && m[x2][y2]==UNFILLED) addmovetoqueue(x2,y2,EMPTY),ok=1;
  }
  return ok;
}

/*  find unfilled squares having two numbers as neighbours */
static int level1twonumberedneighbours() {
  int i,j,k,x2,y2,c,ok=0;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==UNFILLED) {
    for(c=k=0;k<4;k++) {
      x2=i+dx[k],y2=j+dy[k];
      if(x2>=0 && y2>=0 && x2<x && y2<y && m[x2][y2]>0) c++;
    }
    if(c>1) addmovetoqueue(i,j,BLOCKED),ok=1;
  }
  return ok;
}

static int level1enclosedunfilled() {
  int i,j,k,x2,y2,ok=0;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==UNFILLED) {
    for(k=0;k<4;k++) {
      x2=i+dx[k],y2=j+dy[k];
      if(x2>=0 && y2>=0 && x2<x && y2<y && m[x2][y2]!=BLOCKED) goto next;
    }
    addmovetoqueue(i,j,BLOCKED);
    ok=1;
  next:;
  }
  return ok;
}

static int level1enclosefinished() {
  int i,j,tot,num,k,l,x2,y2,r=0;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(!visit[i][j] && m[i][j]>0) {
    genericbfs(i,j,0,&tot,&num,&dummy);
    if(num<1 || num!=tot) goto done;
    for(l=0;l<qe;l+=2) {
      for(k=0;k<4;k++) {
        x2=q[l]+dx[k],y2=q[l+1]+dy[k];
        if(x2<0 || y2<0 || x2>=x || y2>=y) continue;
        if(m[x2][y2]==UNFILLED) addmovetoqueue(x2,y2,BLOCKED),r=1;
      }
    }
  done:
    cleanupbfs();
    if(r) return 1;
  }
  return 0;
}

static int level1hint() {
  if(level1surround1()) return 1;
  if(level12x2()) return 1;
  if(level1twonumberedneighbours()) return 1;
  if(level1enclosedunfilled()) return 1;
  if(level1enclosefinished()) return 1;
  return 0;
}

/*  if there is an unfilled cell which will either:
    - join two islands each containing a numbered cell
    - join two islands, where one has a numbered cell
      and the combination of them exceeds the number
    fill the cell with blocked */
static int level2neighbourtodifferentregions() {
  int i,j,k,res,x2,y2,l;
  int nearid[4],nn;
  markallislandswithid();
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==UNFILLED) {
    for(res=1,nn=0,k=0;k<4;k++) {
      x2=i+dx[k],y2=j+dy[k];
      if(x2>=0 && y2>=0 && x2<x && y2<y && m[x2][y2]>=EMPTY) {
        for(l=0;l<nn;l++) if(nearid[l]==islandmap[x2][y2]) goto neste;
        res+=islandsize[nearid[nn++]=islandmap[x2][y2]];
      }
    neste:;
    }
    if(nn<2) continue;
    /*  try to find either: two numbered islands or one numbered exceeding */
    for(k=0;k<nn-1;k++) for(l=k+1;l<nn;l++) if(islandnum[nearid[k]] || islandnum[nearid[l]]) {
      x2=islandnum[nearid[k]]>islandnum[nearid[l]]?islandnum[nearid[k]]:islandnum[nearid[l]];
      if((islandnum[nearid[k]] && islandnum[nearid[l]]) || (res>x2)) {
        cleanupbfs();
        addmovetoqueue(i,j,BLOCKED);
        return 1;
      }
    }
  }
  cleanupbfs();
  return 0;
}

/*  level 2 (easy): find a region of blocked such that it has only one unfilled
    neighbour and there are other blocked somewhere on the board */
static int level2growblocked() {
  int i,j,k,l,tot,num,qs2,x2,y2,bx,by;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==BLOCKED && !visit[i][j]) {
    qs2=qs;
    genericbfs(i,j,1,&tot,&num,&dummy);
    if(tot==x*y-totalnum) goto done;
    for(bx=by=-1,k=qs2;k<qe;k+=2) for(l=0;l<4;l++) {
      x2=q[k]+dx[l],y2=q[k+1]+dy[l];
      if(x2>=0 && y2>=0 && x2<x && y2<y && m[x2][y2]==UNFILLED) {
        if(bx>-1 && (bx!=x2 || by!=y2)) goto next;  /*  found 2 unfilled neighbours, skip this region */
        else if(bx==-1) bx=x2,by=y2;
      }
    }
    if(bx>-1) {
      cleanupbfs();
      addmovetoqueue(bx,by,BLOCKED);
      return 1;
    }
  next:;
  }
done:
  cleanupbfs();
  return 0;
}

/*  level 2 (easy): find a region of island such that is has only one unfilled
    neighbour and such that the island hasn't reached its size */
static int level2growisland() {
  int i,j,k,l,tot,num,qs2,x2,y2,bx,by;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]>=EMPTY && !visit[i][j]) {
    qs2=qs;
    genericbfs(i,j,0,&tot,&num,&dummy);
    if(num>0 && tot==num) continue;
    for(bx=by=-1,k=qs2;k<qe;k+=2) for(l=0;l<4;l++) {
      x2=q[k]+dx[l],y2=q[k+1]+dy[l];
      if(x2>=0 && y2>=0 && x2<x && y2<y && m[x2][y2]==UNFILLED) {
        if(bx>-1 && (bx!=x2 || by!=y2)) goto next;  /*  found 2 unfilled neighbours, skip this region */
        else if(bx==-1) bx=x2,by=y2;
      }
    }
    if(bx>-1) {
      cleanupbfs();
      addmovetoqueue(bx,by,EMPTY);
      return 1;
    }
  next:;
  }
  cleanupbfs();
  return 0;
}

/*  check if there is an enclosing with n cells, all unfilled and island with one
    numbered cell=n. start search from unfilled cells to avoid finalized islands */
static int level2enclosing() {
  int i,j,k,tot,num,qs2,x2,y2;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(!visit[i][j] && m[i][j]==UNFILLED) {
    qs2=qs;
    genericbfs(i,j,2,&tot,&num,&dummy);
    if(num<1 || tot!=num) continue;
    for(k=qs2;k<qe;k+=2) if(m[x2=q[k]][y2=q[k+1]]==UNFILLED) {
      cleanupbfs();
      addmovetoqueue(x2,y2,EMPTY);
      return 1;
    }
  }
  cleanupbfs();
  return 0;
}

static int level2hint() {
  if(level2neighbourtodifferentregions()) return 1;
  if(level2growblocked()) return 1;
  if(level2growisland()) return 1;
  if(level2enclosing()) return 1;
  return 0;
}

/*  if a region with n-1 cells has two cells to grow to, and an unfilled
    cell is adjacent to both these cells, mark it blocked */
static int level3nminus1() {
  int i,j,k,l,qs2,tot,num,x2,y2,x3,y3,x4,y4;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]>1) {
    qs2=qs;
    genericbfs(i,j,0,&tot,&num,&dummy);
    if(num-1!=tot) continue;
    for(x3=y3=x4=y4=-1,k=qs2;k<qe;k+=2) for(l=0;l<4;l++) {
      x2=q[k]+dx[l],y2=q[k+1]+dy[l];
      if(x2>=0 && y2>=0 && x2<x && y2<y && m[x2][y2]==UNFILLED) {
        if(x3<0) x3=x2,y3=y2;
        else if(x4<0 && (x3!=x2 || y3!=y2)) x4=x2,y4=y2;
        else if((x3!=x2 || y3!=y2) && (x4!=x2 || y4!=y2)) goto next;
      }
    }
    if(x3<0 || x4<0) continue;
    /*  find a cell which borders to both (x3,y3) and (x4,y4) by
        trying all neighbours to (x3,y3) and checking if one of them
        is unfilled and a neighbour of (x4,y4) */
    for(k=0;k<4;k++) {
      x2=x3+dx[k],y2=y3+dy[k];
      if(x2<0 || y2<0 || x2>=x || y2>=y || m[x2][y2]!=UNFILLED) continue;
      if(abs(x2-x4)+abs(y2-y4)==1) {
        cleanupbfs();
        addmovetoqueue(x2,y2,BLOCKED);
        return 1;
      }
    }
  next:;
  }
  cleanupbfs();
  return 0;
}

/*  find all unfilled cells not reachable by any current island. */
/*  warning: slightly complex algorithm ahead:
    at the start, mark all unfilled cells as unreached
    mark all islands with their own id
    for each island with number n and current size m:
      do a bfs with each island square at start point
      (max depth of search: n-m)
      mark each examined square as reached.
      in the bfs, it's not allowed to visit neighbours of other islands.
    at the end of the search, all cells marked as unreached
    are really unreached */
/*  implementation details: need to reset visit array before each
    island bfs. */
static int level3blockunreached() {
  static int reach[MAXS][MAXS];
  int i,steps,j,k,id,tot,num,qe2,cx,cy,x2,y2,l,ok=0,x3,y3;
  markallislandswithid(); /*  generate island id structure */
  for(i=0;i<x;i++) for(j=0;j<y;j++) reach[i][j]=0;  /*  mark as unreachable */
  /*  for each numbered island: do bfs from it */
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]>0) {
    id=islandmap[i][j];
    if(id<0) return 0;
    /*  preliminary bfs to find the cells of the island */
    genericbfs(i,j,0,&tot,&num,&dummy);
    if(num<0 || tot>num) return 0; /*  bail out of solver at any inconsistency */
    /*  reset queue in order to start the search with the entire current queue as
        start seed */
    qs=0;
    steps=num-tot;
    qe2=qe;
    while(steps--) {
      while(qs<qe) {
        cx=q[qs++],cy=q[qs++];
        for(k=0;k<4;k++) {
          x2=cx+dx[k],y2=cy+dy[k];
          if(x2<0 || y2<0 || x2>=x || y2>=y || visit[x2][y2] || m[x2][y2]==BLOCKED) continue;
          /*  check if x2,y2 is adjacent to another numbered island */
          for(l=0;l<4;l++) {
            x3=x2+dx[l],y3=y2+dy[l];
            if(x3<0 || y3<0 || x3>=x || y3>=y || (x2==x3 && y2==y3) || m[x3][y3]<EMPTY) continue;
            if(islandmap[x3][y3]<islandidnum && islandmap[x3][y3]!=id) goto next;
          }
          /*  ok, we can visit square */
          visit[x2][y2]=reach[x2][y2]=1;
          q[qe2++]=x2; q[qe2++]=y2;
        next:;
        }
      }
      qe=qe2;
    }
    cleanupbfs();
  }
  /*  by now, all unvisited cells are non-reachable! set them to blocked */
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==UNFILLED && !reach[i][j]) {
    addmovetoqueue(i,j,BLOCKED),ok=1;
  }
  return ok;
}

/*  find an articulation point for island, which causes one of the following if blocked:
    - there will exist 2x2 blocked
    - there will be an enclosing with disconnected unnumbered island
    - one numbered island with not enough room to grow
    - TODO several numbered islands without combined room to grow?
*/
static int level3articulationpointblank() {
  int i,j,k,x2,y2,tot,num,islands,xx,yy,count,u,v;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==UNFILLED) {
    /*  assume cell is blocked! check for illegal stuff */
    m[i][j]=BLOCKED;
    /*  do bfs, scan each neighbouring area and check for illegal stuff */
    for(k=0;k<4;k++) {
      x2=i+dx[k],y2=j+dy[k];
      if(x2>=0 && y2>=0 && x2<x && y2<y && m[x2][y2]!=BLOCKED) {
        genericbfs(x2,y2,2,&tot,&num,&islands);
        if((num==0 && islands>0) || (num>0 && tot<num)) {
          /*  found an isolated numbered cell, so we have an articulation point
              and the cell must be blank OR we found one numbered cell without
              enough room to grow */
          cleanupbfs();
          m[i][j]=UNFILLED;
          addmovetoqueue(i,j,EMPTY);
          return 1;
        }
        if(!islands) {
          /*  check if there is 2x2 */
          for(xx=0;xx<x-1;xx++) for(yy=0;yy<y-1;yy++) {
            for(count=u=0;u<2;u++) for(v=0;v<2;v++)
              if(m[xx+u][yy+v]==BLOCKED || visit[xx+u][yy+v]) count++;
            if(count>3) {
              /*  found 2x2 blocked */
              cleanupbfs();
              m[i][j]=UNFILLED;
              addmovetoqueue(i,j,EMPTY);
              return 1;
            }
          }
        }
      }
      cleanupbfs();
    }
    m[i][j]=UNFILLED;
  }
  return 0;
}

/*  find an unfinished numbered island which can only grow to exactly its size,
    and finish it. allow it to eat nearby unnumbered islands */
static int level3maximalislandgrow() {
  int i,j,cx,cy,k,x2,y2,tot,num,islands,l,x3,y3,id;
  markallislandswithid();
  /*  try every island */
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]>0) {
    id=islandmap[i][j];
    /*  bfs from start position, only through empty+number */
    genericbfs(i,j,0,&tot,&num,&islands);
    if(islands==num) continue;  /*  already finished */
    if(num<1) return 0; /*  bail out on error */
    qs=0;
    /*  try to expand it */
    while(qs<qe) {
      cx=q[qs++],cy=q[qs++];
      for(k=0;k<4;k++) {
        x2=cx+dx[k],y2=cy+dy[k];
        if(x2<0 || y2<0 || x2>=x || y2>=y || m[x2][y2]==BLOCKED || visit[x2][y2]) continue;
        if(m[x2][y2]==UNFILLED) {
          /*  check if cell borders to another numbered island */
          for(l=0;l<4;l++) {
            x3=x2+dx[l],y3=y2+dy[l];
            if(x3<0 || y3<0 || x3>=x || y3>=y || visit[x3][y3] || m[x3][y3]<EMPTY) continue;
            if(islandmap[x3][y3]>=0 && islandmap[x3][y3]<islandidnum && islandmap[x3][y3]!=id) goto next;
          }
        }
        tot++;
        if(tot>num) goto cancel;
        q[qe++]=x2,q[qe++]=y2;
        visit[x2][y2]=1;
      next:;
      }
    }
    /*  fill it int */
    for(k=0;k<qe;k+=2) if(m[q[k]][q[k+1]]==UNFILLED) {
      addmovetoqueue(q[k],q[k+1],EMPTY);
    }
    cleanupbfs();
    return 1;
  cancel:
    cleanupbfs();
  }
  return 0;
}

static int level3hint() {
  if(level3nminus1()) return 1;
  if(level3blockunreached()) return 1;
  if(level3articulationpointblank()) return 1;
  if(level3maximalislandgrow()) return 1;
  return 0;
}

/*  test every square to see if it is an articulation point for
    blocked: by assuming it is white, check if:
    - there will be a disconnected piece of black */
static int level4articulationpointblocked() {
  int i,j,k,x2,y2,tot,num,res,qs2,l,count;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==UNFILLED) {
    /*  assume cell is empty and check for illegal stuff */
    m[i][j]=EMPTY;
    /*  do bfs from each neighbouring area and check for disconnection */
    for(res=k=0;k<4;k++) {
      x2=i+dx[k],y2=j+dy[k];
      if(x2<0 || y2<0 || x2>=x || y2>=y || (m[x2][y2]!=UNFILLED && m[x2][y2]!=BLOCKED)) continue;
      /*  bfs through blocked and unfilled */
      qs2=qs;
      genericbfs(x2,y2,3,&tot,&num,&dummy);
      /*  check the actual number of blocked encountered */
      for(count=0,l=qs2;l<qe;l+=2) if(m[q[l]][q[l+1]]==BLOCKED) count++;
      if(count && res) {
        /*  found articulation point! */
        cleanupbfs();
        m[i][j]=UNFILLED;
        addmovetoqueue(i,j,BLOCKED);
        return 1;
      }
      if(count) res=count;
    }
    cleanupbfs();
    m[i][j]=UNFILLED;
  }
  return 0;
}

/*  check if an island can grow into a cell such that it blocks the growth of another island.
    then the offending cell must be blocked */
static int level4illegalgrow() {
  int i,j,k,x2,y2,l,id,tot,num,x3,y3,cx,cy,id2,d,found,e;
  markallislandswithid();
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==UNFILLED) {
    /*  bordering to numbered island? */
    /*  to avoid tricky cases, make sure no other island borders to our cell */
    x2=y2=-1;
    for(found=-1,k=0;k<4;k++) {
      x3=i+dx[k],y3=j+dy[k];
      if(x3<0 || y3<0 || x3>=x || y3>=y) continue;
      if(m[x3][y3]>=EMPTY) {
        if((id=islandmap[x3][y3])>=islandidnum) { found=-2; break; }
        if(found<0) found=id;
        else if(found>-1 && found!=id) { found=-2; break; }
        x2=x3,y2=y3;
      }
    }
    if(found<0) continue;
    /*  now, assume that island at (x2,y2) has grown to this cell.
        check if nearby islands cannot grow enough */
    /*  for now, do it the slow way. for each other island, grow it
        (it's permitted to grow through unnumbered islands). */
    id=islandmap[x2][y2];
    m[i][j]=EMPTY; islandmap[i][j]=id;
    /*  manhattan distance heuristic: don't grow island if it's too far away from our change */
    for(k=0;k<x;k++) for(l=0;l<y;l++) if(m[k][l]>0 && (id2=islandmap[k][l])!=id && manhattandist(i,j,k,l)<m[k][l]) {
      genericbfs(k,l,0,&tot,&num,&dummy);
      if(tot==num) goto abortsearch;  /*  ignore completed island */
      qs=0;
      while(qs<qe) {
        cx=q[qs++],cy=q[qs++];
        for(d=0;d<4;d++) {
          x2=cx+dx[d],y2=cy+dy[d];
          if(x2<0 || y2<0 || x2>=x || y2>=y || m[x2][y2]==BLOCKED || visit[x2][y2]) continue;
          /*  reject if other islands are neighbours */
          for(e=0;e<4;e++) {
            x3=x2+dx[e],y3=y2+dy[e];
            if(x3<0 || y3<0 || x3>=x || y3>=y || visit[x3][y3] || m[x3][y3]<EMPTY) continue;
            if(islandmap[x3][y3]!=id2 && islandmap[x3][y3]<islandidnum) goto illegal;
          }
          tot++;
          if(tot>=num) goto abortsearch;  /*  managed to grow the island */
          q[qe++]=x2; q[qe++]=y2;
          visit[x2][y2]=1;
        illegal:;
        }
      }
      /*  found a cell we can't grow into! */
      m[i][j]=UNFILLED; islandmap[i][j]=-1;
      cleanupbfs();
      addmovetoqueue(i,j,BLOCKED);
      return 1;
    abortsearch:
      cleanupbfs();
    }
    m[i][j]=UNFILLED; islandmap[i][j]=-1;
  }
  return 0;
}

/*  find a cell such that if wall is placed, we will block a numbered island's
    growth. then, this cell must be island. */
/*  (this is a partial substitute for island intersection (without combining
    with unnumbered islands), except that this one runs in polynomial time) */
static int level4advancedarticulationisland() {
  int i,j,k,l,tot,num,d,e,x2,y2,cx,cy,x3,y3,id;
  markallislandswithid();
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]==UNFILLED) {
    /*  assume it's blocked */
    m[i][j]=BLOCKED;
    /*  manhattan distance heuristic: don't grow island if it's too far away from our change */
    for(k=0;k<x;k++) for(l=0;l<y;l++) if(m[k][l]>0 && manhattandist(i,j,k,l)<m[k][l]) {
      id=islandmap[k][l];
      genericbfs(k,l,0,&tot,&num,&dummy);
      if(tot==num) goto abortsearch;
      qs=0;
      while(qs<qe) {
        cx=q[qs++],cy=q[qs++];
        for(d=0;d<4;d++) {
          x2=cx+dx[d],y2=cy+dy[d];
          if(x2<0 || y2<0 || x2>=x || y2>=y || visit[x2][y2] || m[x2][y2]==BLOCKED) continue;
          for(e=0;e<4;e++) {
            x3=x2+dx[e],y3=y2+dy[e];
            if(x3<0 || y3<0 || x3>=x || y3>=y || visit[x3][y3] || m[x3][y3]<EMPTY) continue;
            if(islandmap[x3][y3]!=id && islandmap[x3][y3]<islandidnum) goto illegal;
          }
          tot++;
          if(tot>=num) goto abortsearch;
          q[qe++]=x2; q[qe++]=y2;
          visit[x2][y2]=1;
        illegal:;
        }
      }
      /*  can't grow, this cell needs to be blank */
      m[i][j]=UNFILLED;
      cleanupbfs();
      addmovetoqueue(i,j,EMPTY);
      return 1;
    abortsearch:
      cleanupbfs();
    }
    m[i][j]=UNFILLED;
  }
  return 0;
}

static int level4hint() {
  if(level4articulationpointblocked()) return 1;
  if(level4illegalgrow()) return 1;
  if(level4advancedarticulationisland()) return 1;
  return 0;
}

static int lev5bak[MAXS][MAXS];
static int lev5alt1[MAXS][MAXS];
static int lev5alt2[MAXS][MAXS];

/*  copy board from a to b */
static void copyboard(int a[MAXS][MAXS],int b[MAXS][MAXS]) {
  int i,j;
  for(i=0;i<x;i++) for(j=0;j<y;j++) b[i][j]=a[i][j];
}

/*  recalculate st[][] on a fresh board */
static void recalcboard() {
  int i,j;
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]>=EMPTY) recalccompleteness(i,j,0);
  cleanupbfs();
}

/*  do greedy */
/*  return 0 if no faults found, -1 if contradiction */
static int dogreedy(int lev) {
  int r;
  while(1) {
    r=deepverifyboard();
/*    logprintboard();
    logprintf("in loop, result from deepverifyboard: %d\n",r);*/
    if(r<0) return -1;
    if(r>0) return 0;
    if(level1hint()) goto theend;
    if(lev>1 && level2hint()) goto theend;
    if(lev>2 && level3hint()) goto theend;
    if(lev>3 && level4hint()) goto theend;
    break;
  theend:
/*    printf("%d moves in q\n",mqe/3);*/
    silentexecutemovequeue();
  }
  return 0;
}

/*  assume that a cell is white, fill greedily.
    assume that the same cell is black, fill greedily.
    if one of the assumtions results in a contradiction,
    accept the other initial assumption.
    if no contradiction, accept all cells that were filled in similarly.
    only level 1-3 heuristics can be used in greedy.
    parameter lev indicates maximal subheuristic level (max 4) */
static int level5contradiction(int lev) {
  int z,r,l,k,oldsp=getstackpos();
  static int i=0,j=0;
  for(z=0;z<x*y;z++) {
    if(m[i][j]==UNFILLED) {
      /*  assume wall */
      copyboard(m,lev5bak);
      m[i][j]=BLOCKED;
      updatetoscreen(i,j,0);
      r=dogreedy(lev);
      if(r<0) {
        /*  contradiction! */
        copyboard(lev5bak,m);
        recalcboard();
        setstackpos(oldsp);
        addmovetoqueue(i,j,EMPTY);
        return 1;
      }
      copyboard(m,lev5alt1);
      copyboard(lev5bak,m);
      recalcboard();
      setstackpos(oldsp);
      /*  assume blank */
      m[i][j]=EMPTY;
      recalccompleteness(i,j,0);  /*  update st[][] */
      cleanupbfs();
      for(k=0;k<x;k++) for(l=0;l<y;l++) if(visit[k][l]) logprintf("visit not removed at %d,%d\n",k,l);
      r=dogreedy(lev);
      if(r<0) {
        /*  contradiction! */
        copyboard(lev5bak,m);
        recalcboard();
        setstackpos(oldsp);
        addmovetoqueue(i,j,BLOCKED);
        return 1;
      }
      copyboard(m,lev5alt2);
      copyboard(lev5bak,m);
      recalcboard();
      setstackpos(oldsp);
      for(r=k=0;k<x;k++) for(l=0;l<y;l++) if(lev5alt1[k][l]!=UNFILLED && lev5alt1[k][l]==lev5alt2[k][l] && m[k][l]==UNFILLED)
        addmovetoqueue(k,l,lev5alt1[k][l]),r=1;
      if(r) return 1;
    }
    j++;
    if(j==y) {
      j=0,i++;
      if(i==x) i=0;
    }
  }
  return 0;
}

static int level5hint() {
  if(level5contradiction(3)) return 1;
  if(level5contradiction(4)) return 1;
  return 0;
}

static int hint() {
  if(verifyboard()<0) return -1;
  if(level1hint()) return 1;
  if(level2hint()) return 1;
  if(level3hint()) return 1;
  if(level4hint()) return 1;
  if(level5hint()) return 1;
  return 0;
}

static uchar colarray[]={
  1,2,3,12,18,96,78
};
#define COLSIZE 7

static void showverify() {
  int colour=0,qs2,tot,num,k,i,j;
  if(SDL_MUSTLOCK(screen)) SDL_LockSurface(screen);
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(!visit[i][j] && m[i][j]==BLOCKED) {
    qs2=qs;
    genericbfs(i,j,1,&tot,&num,&dummy);
    for(k=qs2;k<qe;k+=2) drawsolidcell8(q[k],q[k+1],colarray[colour%COLSIZE]);
    colour++;
  }
  /*  display 2x2 blocked */
  for(i=0;i<x-1;i++) for(j=0;j<y-1;j++) if(m[i][j]==BLOCKED && m[i+1][j]==BLOCKED && m[i][j+1]==BLOCKED && m[i+1][j+1]==BLOCKED) {
    drawsolidcell8(i,j,DARKRED8);
    drawsolidcell8(i,j+1,DARKRED8);
    drawsolidcell8(i+1,j,DARKRED8);
    drawsolidcell8(i+1,j+1,DARKRED8);
  }
  if(SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
  SDL_UpdateRect(screen,0,0,resx,resy);
  cleanupbfs();
  anykeypress();
  /*  display normal level */
  if(SDL_MUSTLOCK(screen)) SDL_LockSurface(screen);
  for(i=0;i<x;i++) for(j=0;j<y;j++) updatecell(i,j);
  if(SDL_MUSTLOCK(screen)) SDL_UnlockSurface(screen);
  SDL_UpdateRect(screen,0,0,resx,resy);
}
#undef COLSIZE

static void processkeydown(int key) {
  int res;
  if(key==undokey) undo();
  else if(key==hintkey) {
    if(!executeonemovefromqueue(1)) {
      res=hint();
      if(res>0) executeonemovefromqueue(1);
      else if(!res) messagebox("Sorry, no moves found.");
      else messagebox("Sorry, hint will not work on an illegal board.");
    }
  } else if(key==SDLK_j) {  /*  temporary: superhintkey */
    res=hint();
    if(res>0) {
      executemovequeue();
      while(hint()>0) executemovequeue();
      if(verifyboard()<1) messagebox("Sorry, no more moves found.");
    } else if(!res) messagebox("Sorry, no moves found.");
    else messagebox("Sorry, hint will not work on an illegal board.");
  } else if(key==verifykey) showverify();
}

void nurikabe(char *path) {
  int event,i,j;
  loadpuzzle(path);
  initcompleteness();
  for(i=0;i<x;i++) for(j=0;j<y;j++) if(m[i][j]>0) recalccompleteness(i,j,1);
  cleanupbfs();
  drawgrid();
  do {
    event=getevent();
    switch(event) {
    case EVENT_RESIZE:
      drawgrid();
    case EVENT_NOEVENT:
      break;
    case EVENT_MOUSEDOWN:
      processmousedown();
      if(verifyboard()==1) {
        messagebox("You are winner!");
        return;
      }
      break;
    default:
      /*  catch intervals of values here */
      if(event>=EVENT_KEYDOWN && event<EVENT_KEYUP) {
        processkeydown(event-EVENT_KEYDOWN);
        if(verifyboard()==1) {
          messagebox("You are winner!");
          return;
        }
      }
    }
  } while(event!=EVENT_QUIT && !keys[SDLK_ESCAPE]);
}
